unit IECommThread;

interface

uses
  Classes, StdCtrls, SyncObjs, SysUtils;

type
  TIECommThread = class(TThread)
  { Purpose: This thread object acts as a go-between for the IE threads and the
    main thread.  It provides a message queue for the thread to avoid blocking
    the main thread.  This thread should be passed to the IEThread. }
  private
    { Private declarations }
    FListBox: TListBox;
    FQueue: TStrings;
    FText: String;
    FCriticalSection: TCriticalSection;
  protected
    procedure Execute; override;
    procedure Pop;
    property Queue: TStrings read FQueue;
    property CriticalSection: TCriticalSection read FCriticalSection;
    property Text: String read FText;
  public
    constructor Create( ListBox: TListBox ); virtual;
    destructor Destroy; override;
    procedure Push( Text: String );
    property ListBox: TListBox read FListBox;
  end;

implementation

{ Important: Methods and properties of objects in VCL can only be used in a
  method called using Synchronize, for example,

      Synchronize(UpdateCaption);

  and UpdateCaption could look like,

    procedure TIECommThread.UpdateCaption;
    begin
      Form1.Caption := 'Updated in a thread';
    end; }

{ TIECommThread }

constructor TIECommThread.Create(ListBox: TListBox);
begin
     inherited Create( False );
     Priority := tpLower;
     FCriticalSection := TCriticalSection.Create;
     FQueue := TStringList.Create;
     FListBox := ListBox;
end;

destructor TIECommThread.Destroy;
begin
     FCriticalSection.Free;
     FQueue.Free;
     inherited;
end;

procedure TIECommThread.Execute;
var
   SomethingToSend: Boolean;
begin
     while not Terminated do
     begin
          SomethingToSend := False;
          // Get a string from the queue, protect access
          CriticalSection.Enter;
          try
             if Queue.Count > 0 then
             begin
                  FText := Trim(Queue[0]);
                  Queue.Delete(0);
                  SomethingToSend := True;
             end;
          finally
             CriticalSection.Leave;
          end;
          if SomethingToSend then
             Synchronize(Pop);
     end;
end;

procedure TIECommThread.Push(Text: String);
begin
     // protect access to the Queue
     CriticalSection.Enter;
     try
        Queue.Add( Text );
     finally
        CriticalSection.Leave;
     end;
end;

procedure TIECommThread.Pop;
begin
     ListBox.Items.Add( Text );
end;

end.
